'use strict';

/* --------------------------------------------------------------
 cookies_notice_controller.js 2016-09-09
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2016 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Cookie Notice Controller
 *
 * Compatibility module that handles the "Cookie Notice" page under the "Rights" menu of "Shop Settings" section.
 * The data of the form are updated upon change and this module will just post them to LawsController. Check out
 * the fields that are language dependent, they will be changed when the user selects a language from the language
 * switcher component.
 *
 * @module Compatibility/cookie_notice_controller
 */
gx.compatibility.module('cookie_notice_controller', ['loading_spinner', gx.source + '/libs/editor_values', gx.source + '/libs/info_messages'], function (data) {

    'use strict';

    // --------------------------------------------------------------------
    // VARIABLES
    // --------------------------------------------------------------------

    var
    /**
     * Module Selector
     *
     * @type {object}
     */
    $this = $(this),


    /**
     * Module Instance
     *
     * @type {object}
     */
    module = {
        model: {
            formData: jse.core.config.get('appUrl') + '/admin/admin.php?do=Laws/GetCookiePreferences&pageToken=' + jse.core.config.get('pageToken')
        }
    };

    // --------------------------------------------------------------------
    // FUNCTIONS
    // --------------------------------------------------------------------

    /**
     * On Form Submit Event
     *
     * @param {object} event jQuery Event Object
     */
    var _onFormSubmit = function _onFormSubmit(event) {
        event.preventDefault();

        // Prepare form data and send them to the LawsController class. 
        var postUrl = jse.core.config.get('appUrl') + '/admin/admin.php?do=Laws/SaveCookiePreferences',
            postData = $.extend({ pageToken: jse.core.config.get('pageToken') }, module.model.formData),
            $spinner;

        $.ajax({
            url: postUrl,
            type: 'POST',
            data: postData,
            dataType: 'json',
            beforeSend: function beforeSend() {
                $spinner = jse.libs.loading_spinner.show($this, '4');
            }
        }).done(function () {
            // Display success message.
            jse.libs.info_messages.addSuccess(jse.core.lang.translate('TXT_SAVE_SUCCESS', 'admin_general'));
        }).fail(function (jqxhr, textStatus, errorThrown) {
            // Display failure message.
            jse.libs.info_messages.addError(jse.core.lang.translate('TXT_SAVE_ERROR', 'admin_general'));
            jse.core.debug.error('Could not save Cookie Notice preferences.', jqxhr, textStatus, errorThrown);
        }).always(function () {
            jse.libs.loading_spinner.hide($spinner);

            // Scroll to the top, so that the user sees the appropriate message.
            $('html, body').animate({ scrollTop: 0 });
        });
    };

    /**
     * On Language Flag Click Event
     *
     * @param {object} event jQuery event object.
     */
    var _onLanguageClick = function _onLanguageClick(event) {
        event.preventDefault();

        $(this).siblings().removeClass('active');
        $(this).addClass('active');

        // Load the language specific fields.
        $.each(module.model.formData, function (name, value) {
            var $element = $this.find('[name="' + name + '"]');

            if ($element.data('multilingual') !== undefined) {
                var selectedLanguageCode = $('.languages a.active').data('code');
                $element.val(value[selectedLanguageCode]);
                if ($element.is('textarea') && $element.parents('.editor-wrapper').length) {
                    jse.libs.editor_values.setValue($element, value[selectedLanguageCode]);
                }
            } else {
                $element.val(value);

                if ($element.is(':checkbox') && value === 'true') {
                    $element.parent().addClass('checked');
                    $element.prop('checked', true);
                }

                if (name === 'position' && !value) {
                    $element.find('option[value="top"]').prop('selected', true).trigger('change');
                }
            }
        });
    };

    /**
     * On Input Element Change Event
     */
    var _onInputChange = function _onInputChange() {
        var $element = $(this);

        if ($element.data('multilingual') !== undefined) {
            var selectedLanguageCode = $('.languages a.active').data('code');
            module.model.formData[$element.attr('name')][selectedLanguageCode] = $element.val();
        } else {
            module.model.formData[$element.attr('name')] = $element.val();
        }
    };

    /**
     * On Switcher Widget Click Event
     */
    var _onSwitcherClick = function _onSwitcherClick() {
        module.model.formData[$(this).find('input:checkbox').attr('name')] = $(this).hasClass('checked');
    };

    // --------------------------------------------------------------------
    // INITIALIZATION
    // --------------------------------------------------------------------

    module.init = function (done) {
        // Bind form event handlers. 
        $this.on('submit', _onFormSubmit).on('click', '.languages a', _onLanguageClick).on('click', '.switcher', _onSwitcherClick);

        $this.find('input:hidden, input:text, select, textarea').on('change', _onInputChange);

        // Select active language.
        $('.languages').find('.active').trigger('click');

        // Set the color-preview colors.
        $this.find('.color-preview').each(function () {
            $(this).css('background-color', $(this).siblings('input:hidden').val());
        });

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
